/**
* \file: WaylandConnection.cpp
*
* \version: 0.4
*
* \release: $Name:$
*
* Wayland utilization for SPI related use cases
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string>
#include <cstring>
#include <cerrno>
#include <poll.h>
#include <unistd.h>
#include <sys/prctl.h>
#include <sys/eventfd.h>
#include <uspi/ITouchFacade.h>
#include "WaylandConnection.h"

namespace adit { namespace uspi {
using std::move;
using std::unique_ptr;
using std::to_string;
using std::strerror;

struct wl_registry_listener WaylandConnection::registryListener =
{
    WaylandConnection::onRegistryGlobal,
    WaylandConnection::onRegistryGlobalRemove
};

// todo seat related stuff including touch and pointer can go to the WlSeat class
struct wl_seat_listener WaylandConnection::seatListener =
{
    WaylandConnection::onSeatCapabilities,
    WaylandConnection::onSeatName
};

WaylandConnection::WaylandConnection(ITouchFacadeCallbacks& inCallbacks, int inDisplayWidth, int inDisplayHeight,
        int inMaxFingerNum)
: mCallbacks(inCallbacks), mDisplayWidth(inDisplayWidth), mDisplayHeight(inDisplayHeight), mWlDisplay(nullptr),
  mInputQueue(nullptr), mWlRegistry(nullptr), mInputThreadRunning(false), mInputThread(0), shutDownEventFd(-1)
{
    mTouchListenerObj = move(unique_ptr<WaylandTouchListener>(new WaylandTouchListener(mCallbacks, mDisplayWidth,
            mDisplayHeight, inMaxFingerNum)));
    mWlTouchListener = &WaylandTouchListener::mWlTouchListener;

    mPointerListenerObj = move(unique_ptr<WaylandPointerListener>(new WaylandPointerListener(mCallbacks, mDisplayWidth,
            mDisplayHeight)));
    mWlPointerListener = &WaylandPointerListener::mWlPointerListener;

    mSeatListMutex = PTHREAD_MUTEX_INITIALIZER;
}

WaylandConnection::~WaylandConnection()
{
    if (!stop())
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Destruction cannot stop WaylandConnection");
    }
}

bool WaylandConnection::start(wl_display* inWlDisplay)
{
    int wl_error = -1;

    mWlDisplay = inWlDisplay;

    mWlRegistry = wl_display_get_registry(mWlDisplay);
    if (mWlRegistry == nullptr)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland Connection failed to get registry");
        return false;
    }

    wl_error = wl_registry_add_listener(mWlRegistry, &registryListener, this);
    if (wl_error < 0)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland Connection failed to set registry listener: "
                + to_string(wl_error));
        return false;
    }

    mInputQueue = wl_display_create_queue(mWlDisplay);
    if (mInputQueue == nullptr)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland Connection failed to create a queue for input events");
        return false;
    } else {
        /* Registry events will come to inputQueue */
        wl_proxy_set_queue((wl_proxy*) mWlRegistry, mInputQueue);
    }

    wl_error = wl_display_roundtrip_queue(mWlDisplay, mInputQueue);
    if (wl_error < 0)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland round trip to get Registry Global event failed: "
                + to_string(wl_error));
        return false;
    }

    wl_error = wl_display_roundtrip_queue(mWlDisplay, mInputQueue);
    if (wl_error < 0)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland round trip to get Seat Capabilities event failed: "
                + to_string(wl_error));
        return false;
    }

    if (!startInputThread())
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Could not start the Input thread");
        return false;
    }

    return true;
}

bool WaylandConnection::stop()
{
    if(stopInputThread() == false)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland Connection cannot stop input thread");
        return false;
    }

    int err = pthread_mutex_lock(&mSeatListMutex);
    if (err == 0)
    {
        for (auto it = mSeatList.begin(); it != mSeatList.end(); it++)
        {
            delete (*it);
        }
        mSeatList.clear();

        err = pthread_mutex_unlock(&mSeatListMutex);
        if (err != 0)
        {
            mCallbacks.onLogging(USPI_LOG_ERROR, "Stop: cannot unlock the seat list, err: " + to_string(err));
            return false;
        }

    } else {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Cannot access the seats to destroy them, err: " + to_string(err));
        return false;
    }

    uspi_safe_call(wl_event_queue_destroy, mInputQueue);
    uspi_safe_call(wl_registry_destroy, mWlRegistry);

    return true;
}

void WaylandConnection::setLateResolution(int inWidth, int inHeight)
{
    mDisplayWidth = inWidth;
    mDisplayHeight = inHeight;

    mTouchListenerObj->setLateResolution(mDisplayWidth, mDisplayHeight);
    mPointerListenerObj->setLateResolution(mDisplayWidth, mDisplayHeight);
}

void WaylandConnection::addWlSeat(WaylandSeat* inSeat)
{
    int err = pthread_mutex_lock(&mSeatListMutex);
    if (err == 0)
    {
        mSeatList.push_back(inSeat);

        err = pthread_mutex_unlock(&mSeatListMutex);
        if (err != 0)
        {
            mCallbacks.onLogging(USPI_LOG_ERROR, "addWlSeat cannot unlock the seat list, err: " + to_string(err));
        }

    } else {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Cannot access the seat list to add a new one, err: " + to_string(err));
    }
}

bool WaylandConnection::startInputThread()
{
    if (mInputThreadRunning)
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Input thread is already running");
        return false;
    }

    /* create eventFd to send stop event */
    if (0 > (shutDownEventFd = eventfd(0, 0)))
    {
        mCallbacks.onLogging(USPI_LOG_ERROR, "Failed to create shutdown eventFd, error: "
                + to_string(errno) + " " + strerror(errno));
        return false;
    }

    mInputThreadRunning = true;

    int ret = pthread_create(&mInputThread, NULL, inputThread, this);
    if (ret != 0)
    {
        mInputThreadRunning = false;
        mCallbacks.onLogging(USPI_LOG_ERROR, "Input thread: pthread_create failed with error: " + to_string(ret));
        return false;
    }
    return true;
}

bool WaylandConnection::stopInputThread()
{
    if (mInputThreadRunning)
    {
        mInputThreadRunning = false;

        if (shutDownEventFd >= 0) {
            /* trigger shutdown event */
            if (eventfd_write(shutDownEventFd, shutDownEvent) != 0)
            {
                mCallbacks.onLogging(USPI_LOG_ERROR, "Failed to trigger input thread shutdown event");
                return false;
            }
        } else {
            mCallbacks.onLogging(USPI_LOG_ERROR, "shutDownEventFd is invalid");
            return false;
        }

        int ret = pthread_join(mInputThread, nullptr);
        if (ret != 0)
        {
            /* if pthread_join fails just log the error using errno */
            mCallbacks.onLogging(USPI_LOG_ERROR, "pthread_join failed with error: " + to_string(ret));
            return false;
        }

        if (shutDownEventFd >= 0) {
            if (0 != close(shutDownEventFd))
            {
                mCallbacks.onLogging(USPI_LOG_ERROR, "close shutDownEventFd failed with error: "
                        + to_string(errno) + " " + strerror(errno));
                return false;
            }
            shutDownEventFd = -1;
        }
    }
    return true;
}

void* WaylandConnection::inputThread(void* inMe)
{
    uspi_return_value_on_invalid_argument(uspi, inMe == nullptr, nullptr);

    auto me = static_cast<WaylandConnection*>(inMe);

    /* set thread name */
    prctl(PR_SET_NAME, "WaylandTouch", 0, 0, 0);

    me->mCallbacks.onLogging(USPI_LOG_DEBUG, "Input thread started");

    while (me->mInputThreadRunning)
    {
        int error = -1;

        while (wl_display_prepare_read_queue(me->mWlDisplay, me->mInputQueue) != 0)
        {
            /* dispatch events which might be read already */
            error = wl_display_dispatch_queue_pending(me->mWlDisplay, me->mInputQueue);
            if (error < 0)
            {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Failed dispatching events which are already read, with error: "
                        + to_string(errno) + " " + strerror(errno) + " - Exit input thread");
                return nullptr;
            }
        }

        error = wl_display_flush(me->mWlDisplay);
        if (error < 0)
        {
            me->mCallbacks.onLogging(USPI_LOG_ERROR, "Wayland flush failed with error: "
                    + to_string(errno) + " " + strerror(errno) + " - Exit input thread");
            wl_display_cancel_read(me->mWlDisplay);
            return nullptr;
        }

        /* poll for input events */
        pollfd pfd[2];
        /* shutdown events */
        pfd[0].fd = me->shutDownEventFd;
        pfd[0].events = me->shutDownEvent;
        /* input events */
        pfd[1].fd = wl_display_get_fd(me->mWlDisplay);
        pfd[1].events = POLLIN;

        int retval = poll(pfd, 2, -1);
        if(retval <= 0)
        {
            wl_display_cancel_read(me->mWlDisplay);

        /* handle shut down event before Wayland event */
        } else if (pfd[0].revents & me->shutDownEvent) {

            wl_display_cancel_read(me->mWlDisplay);
            break;

        } else if (pfd[1].revents & POLLIN) {

            error = wl_display_read_events(me->mWlDisplay);
            if (error < 0)
            {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Reading wayland event failed with error: "
                        + to_string(errno) + " " + strerror(errno));
                /* todo define uspi error codes */
                me->mCallbacks.onTouchError(7321);
            }

            error = wl_display_dispatch_queue_pending(me->mWlDisplay, me->mInputQueue);
            if (error < 0)
            {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Dispatching the input queue failed with error: "
                        + to_string(errno) + " " + strerror(errno));
                /* todo define uspi error codes */
                me->mCallbacks.onTouchError(4893);
            }
        }
    }
    return nullptr;
}

void WaylandConnection::onRegistryGlobal(void* inMe, wl_registry* inRegistry, uint32_t inName, const char* inInterface,
        uint32_t inVersion)
{
    uspi_unused_variable(inVersion);
    uspi_return_on_invalid_argument(uspi, inMe == nullptr);
    uspi_return_on_invalid_argument(uspi, inRegistry == nullptr);
    uspi_return_on_invalid_argument(uspi, inInterface == nullptr);

    auto me = static_cast<WaylandConnection*>(inMe);

    if (!strcmp(inInterface, "wl_seat"))
    {
        WaylandSeat* wl_seat_obj = new WaylandSeat();
        if (wl_seat_obj)
        {
            auto wl_seat = static_cast<struct wl_seat*>(wl_registry_bind(inRegistry, inName, &wl_seat_interface, 3));

            if (wl_seat)
            {
                int error = wl_seat_add_listener(wl_seat, &me->seatListener, wl_seat_obj);
                if (error < 0)
                {
                    me->mCallbacks.onLogging(USPI_LOG_ERROR, "Registry Global: Failed to add Seat Listener with error: "
                            + to_string(error));
                }
                /* Seat events will come to inputQueue */
                wl_proxy_set_queue((wl_proxy*) wl_seat, me->mInputQueue);

                wl_seat_obj->setWlSeat(wl_seat);
                wl_seat_obj->setWaylandConnection(me);
                wl_seat_obj->setSeatId(inName);
                me->addWlSeat(wl_seat_obj);

            } else {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Registry Global: Failed to bind Wayland Seat Interface");
            }

        } else {
            me->mCallbacks.onLogging(USPI_LOG_ERROR, "Registry Global: Cannot create new WaylandSeat object");
        }
    }
}

void WaylandConnection::onRegistryGlobalRemove(void* inMe, wl_registry* inRegistry, uint32_t inName)
{
    uspi_unused_variable(inRegistry);
    uspi_return_on_invalid_argument(uspi, inMe == nullptr);

    auto me = static_cast<WaylandConnection*> (inMe);

    int err = pthread_mutex_lock(&(me->mSeatListMutex));
    if (err == 0)
    {
        for (auto it = me->mSeatList.begin(); it != me->mSeatList.end(); it++)
        {
            if ((*it)->seatId() == inName )
            {
                me->mCallbacks.onLogging(USPI_LOG_INFO, "Registry Global Remove seat " + to_string(inName));
                delete (*it);
                me->mSeatList.erase(it);
                break;
            }
        }

        err = pthread_mutex_unlock(&(me->mSeatListMutex));
        if (err != 0)
        {
            me->mCallbacks.onLogging(USPI_LOG_ERROR, "Registry Global Remove: cannot unlock the seat list, err: "
                    + to_string(err));
        }

    } else {
        me->mCallbacks.onLogging(USPI_LOG_ERROR, "Registry Global Remove: cannot access the seat list to remove one, err: "
                + to_string(err));
    }
}

void WaylandConnection::onSeatCapabilities(void* inData, wl_seat* inSeat, uint32_t inCapabilities)
{
    uspi_unused_variable(inSeat);
    uspi_return_on_invalid_argument(uspi, inData == nullptr);

    auto wl_seat_obj = static_cast<WaylandSeat*> (inData);

    auto me = wl_seat_obj->waylandConnection();
    if (!me)
    {
        /* Seat object has no Wayland connection */
        return;
    }

    auto wl_seat = wl_seat_obj->wlSeat();
    if (!wl_seat) {
        me->mCallbacks.onLogging(USPI_LOG_ERROR, "Seat object has no wayland_seat");
        return;
    }

    /* handle possible touch capability */
    if (me->mWlTouchListener && (inCapabilities & WL_SEAT_CAPABILITY_TOUCH))
    {
        /* if wayland touch not yet set */
        auto wl_touch = wl_seat_obj->wlTouch();
        if (!wl_touch)
        {
            wl_touch = wl_seat_get_touch(wl_seat);
            if (wl_touch)
            {
                wl_touch_set_user_data(wl_touch, (void*)(me->mTouchListenerObj.get()));

                int error = wl_touch_add_listener(wl_touch, me->mWlTouchListener, (void*)me->mTouchListenerObj.get());
                if (error < 0)
                {
                    uspi_safe_call(wl_touch_release, wl_touch);

                    me->mCallbacks.onLogging(USPI_LOG_ERROR, "Failed to add Touch Listener");
                }

            } else {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Could not get wl_touch from wl_seat");
            }

        } else {
            me->mCallbacks.onLogging(USPI_LOG_WARN, "Touch is already set for this seat");
        }

        wl_seat_obj->setWlTouch(wl_touch);

    } else {
        if (nullptr == me->mWlTouchListener)
        {
            me->mCallbacks.onLogging(USPI_LOG_DEBUG, "No Touch Listener to register to Wayland Seat");

        } else {
            me->mCallbacks.onLogging(USPI_LOG_DEBUG, "Seat does not support touch");
        }
    }

    /* handle possible pointer capability */
    if (me->mWlPointerListener && (inCapabilities & WL_SEAT_CAPABILITY_POINTER))
    {
        /* if wayland pointer not yet set */
        auto wl_pointer = wl_seat_obj->wlPointer();
        if (!wl_pointer)
        {
            wl_pointer = wl_seat_get_pointer(wl_seat);
            if (wl_pointer)
            {
                wl_pointer_set_user_data(wl_pointer, (void*)(me->mPointerListenerObj.get()));

                int error = wl_pointer_add_listener(wl_pointer, me->mWlPointerListener,
                        (void*)me->mPointerListenerObj.get());
                if (error < 0)
                {
                    uspi_safe_call(wl_pointer_release, wl_pointer);

                    me->mCallbacks.onLogging(USPI_LOG_ERROR, "Failed to add Pointer Listener");
                }

            } else {
                me->mCallbacks.onLogging(USPI_LOG_ERROR, "Could not get wl_pointer from wl_seat");
            }

        } else {

            me->mCallbacks.onLogging(USPI_LOG_WARN, "Pointer is already set for this seat");
        }

        wl_seat_obj->setWlPointer(wl_pointer);

    } else {
        if (nullptr == me->mWlPointerListener)
        {
            me->mCallbacks.onLogging(USPI_LOG_DEBUG, "No Pointer Listener to register to Wayland Seat");

        } else {
            me->mCallbacks.onLogging(USPI_LOG_DEBUG, "Seat does not support pointer");
        }
    }

    /* ignore keyboard capability */

}

void WaylandConnection::onSeatName(void* inData, wl_seat* inSeat, const char *inName)
{
    uspi_unused_variable(inSeat);
    uspi_return_on_invalid_argument(uspi, inData == nullptr);
    uspi_return_on_invalid_argument(uspi, inName == nullptr);

    auto wl_seat_obj_ptr = static_cast<WaylandSeat*> (inData);
    wl_seat_obj_ptr->setSeatName(inName);
    auto me = wl_seat_obj_ptr->waylandConnection();
    if (me)
    {
        me->mCallbacks.onLogging(USPI_LOG_INFO, "Seat event for name " + std::string(inName));
    }
}

} } /* namespace adit { namespace uspi { */
